<?php
// round_live.php (CSV-driven)
declare(strict_types=1);
date_default_timezone_set('Europe/Amsterdam');

ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);


function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function normalizeWs(string $s): string {
  $s = trim($s);
  $s = preg_replace('/\s+/u', ' ', $s);
  return $s ?? trim($s);
}

function splitMatch(string $wedstrijd): array {
  $w = normalizeWs($wedstrijd);

  // toleranter splitten op " vs "
  if (preg_match('/^(.*?)\s+vs\s+(.*?)$/iu', $w, $m)) {
    return [trim($m[1]), trim($m[2])];
  }
  // fallback
  return [$w, ''];
}

function detectType(string $text): string {
  $lower = mb_strtolower($text, 'UTF-8');

  if (mb_strpos($lower, 'gele kaart') !== false) return 'card_y';
  if (mb_strpos($lower, 'rode kaart') !== false) return 'card_r';
  if (mb_strpos($lower, 'wissel') !== false) return 'sub';
  if (mb_strpos($lower, 'bles') === 0 || mb_strpos($lower, 'bles.') === 0) return 'inj';
  if (preg_match('/^\d+\s*-\s*\d+\s*:/u', $text)) return 'goal';

  return 'note';
}

$showNuLiveFeeds = ['all','eredivisie','eerstedivisie','tweededivisie'];

/**
 * CSV format:
 * Minuut_Event ; Wedstrijd
 * - Minuut_Event can be: "13e min : ...", "Aantal toeschouwers : 22537"
 */
function parseRoundCsv(string $csvFile): array {
  if (!is_file($csvFile)) return [];

  $fh = fopen($csvFile, 'r');
  if (!$fh) return [];

  // probeer header te lezen; als het geen header is, gebruiken we 'm als data
  $first = fgetcsv($fh, 0, ';');
  if ($first === false) { fclose($fh); return []; }

  $rows = [];
  $looksLikeHeader = false;
  if (count($first) >= 2) {
    $a = mb_strtolower(trim((string)$first[0]));
    $b = mb_strtolower(trim((string)$first[1]));
    if (str_contains($a, 'minuut') && str_contains($b, 'wedstrijd')) $looksLikeHeader = true;
  }
  if (!$looksLikeHeader) $rows[] = $first;

  while (($r = fgetcsv($fh, 0, ';')) !== false) {
    if (count($r) < 2) continue;
    $rows[] = $r;
  }
  fclose($fh);

  // group by wedstrijd
  $byMatch = [];

  foreach ($rows as $r) {
    $minEvt = normalizeWs((string)$r[0]);
$wed    = normalizeWs((string)$r[1]);

if ($minEvt === '' || $wed === '') continue;

// alleen echte wedstrijden
if (!preg_match('/\s+vs\s+/iu', $wed)) continue;

    if (!isset($byMatch[$wed])) {
      [$home, $away] = splitMatch($wed);
      $byMatch[$wed] = [
        'home' => $home,
        'away' => $away,
        'attendance' => null,
        'events' => []
      ];
    }

    // Attendance regel
    if (preg_match('/^Aantal toeschouwers\s*:\s*(\d+)/iu', $minEvt, $m)) {
      $byMatch[$wed]['attendance'] = (int)$m[1];
      continue;
    }

    // Event regel "13e min : ..."
    if (preg_match('/^(\d+)\s*e\s*min\s*:\s*(.+)$/iu', $minEvt, $m)) {
      $min  = (int)$m[1];
      $text = trim($m[2]);
      $type = detectType($text);

      $byMatch[$wed]['events'][] = [
        'min'  => $min,
        'type' => $type,
        'text' => $text
      ];
    }
  }

  // final: sort events + return indexed array
  $out = array_values($byMatch);
  foreach ($out as &$match) {
    usort($match['events'], fn($a,$b)=>($a['min'] <=> $b['min']) ?: strcmp($a['text'], $b['text']));
  }
  unset($match);

  return $out;
}

// ====== DATA (selecteer CSV via ?feed=all|eredivisie|eerstedivisie|tweededivisie) ======
$feeds = [
  'all'          => ['label' => 'Alles',          'file' => 'livefeed/raw_live_all.csv'],
  'eredivisie'   => ['label' => 'Eredivisie',     'file' => 'livefeed/raw_live_eredivisie.csv'],
  'eerstedivisie'=> ['label' => 'Eerste Divisie', 'file' => 'livefeed/raw_live_eerstedivisie.csv'],
  'tweededivisie'=> ['label' => 'Tweede Divisie', 'file' => 'livefeed/raw_live_tweededivisie.csv'],
];

// backward compat: als iemand nog raw_live.csv heeft, dan “all” daarop laten vallen
$legacy = __DIR__ . '/raw_live.csv';
if (is_file($legacy) && !is_file(__DIR__ . '/raw_live_all.csv')) {
  $feeds['all']['file'] = 'raw_live.csv';
}

$feedKey = $_GET['feed'] ?? 'all';
if (!isset($feeds[$feedKey])) $feedKey = 'all';

$csvFile = __DIR__ . '/' . $feeds[$feedKey]['file'];
$feedLabel = $feeds[$feedKey]['label'];

// ====== Nu Live (tekstbestand) ======
$nuLivePath = __DIR__ . '/livefeed/DezeRonde.txt';
$nuLiveText = '';

$leagueName = 'Panenka League'; // fallback

if (is_file($nuLivePath)) {
  $lines = file($nuLivePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  foreach ($lines as $line) {
    if (preg_match('/^\s*LeagueNaam\s*:\s*(.+)$/i', $line, $m)) {
      $leagueName = trim($m[1]);
      break;
    }
  }
}

if (is_file($nuLivePath)) {
  $nuLiveText = trim((string)file_get_contents($nuLivePath));
  // maak het 1 regel (optioneel)
  $nuLiveText = preg_replace("/\r\n|\r|\n/", " • ", $nuLiveText) ?? $nuLiveText;
}

// ====== Stadionnamen (Teams;Stadion) ======
$stadionPath = __DIR__ . '/livefeed/stadionnamen.csv';
$stadionByTeam = [];

if (is_file($stadionPath)) {
  if (($fh = fopen($stadionPath, 'r')) !== false) {
    // header weg (Teams;Stadion)
    $first = fgetcsv($fh, 0, ';');

    while (($r = fgetcsv($fh, 0, ';')) !== false) {
      if (count($r) < 2) continue;
      $team = normalizeWs((string)$r[0]);
      $stad = normalizeWs((string)$r[1]);
      if ($team === '' || $stad === '') continue;
      $stadionByTeam[$team] = $stad;
    }
    fclose($fh);
  }
}



$round = parseRoundCsv($csvFile);
$hasData = count($round) > 0;

// stadionnamen koppelen aan wedstrijden
foreach ($round as &$m) {
  $m['stadium_home'] = $stadionByTeam[$m['home']] ?? '';
  $m['stadium_away'] = $stadionByTeam[$m['away']] ?? '';
  // voorkeur: thuisstadion tonen, fallback op uit
  $m['stadium'] = $m['stadium_home'] !== '' ? $m['stadium_home'] : $m['stadium_away'];
}
unset($m);


$speedMs = 1500;
?>
<!doctype html>
<html lang="nl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title><?= h($leagueName) ?> — Speelronde Live</title>
  <style>
    :root{
  --bg:#07070b; --panel:#0f1016; --line:#2a2d3a;
  --text:#e9ecff; --muted:#9aa3c7; --accent:#a7ff1a;
  --warn:#ffd000; --danger:#ff3b3b; --shadow: rgba(0,0,0,.45);
}

*{box-sizing:border-box}

body{
  margin:0;
  background: radial-gradient(900px 600px at 20% 0%, rgba(167,255,26,.08), transparent 60%),
              radial-gradient(900px 600px at 80% 10%, rgba(120,110,255,.08), transparent 60%),
              var(--bg);
  color:var(--text);
  font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
}

.wrap{max-width:1200px;margin:0 auto;padding:18px 14px 32px}

/* ===== Topbar layout (3 rijen) ===== */
.topbar{
  display:block;
  padding:14px;
  border:1px solid var(--line);
  border-radius:14px;
  background: linear-gradient(180deg, rgba(255,255,255,.05), rgba(255,255,255,.02));
  box-shadow: 0 10px 30px var(--shadow);
}

.toprow{
  display:flex;
  align-items:flex-start;
  justify-content:space-between;
  gap:12px;
}

.brand{display:flex;flex-direction:column;gap:2px; flex:1 1 auto; min-width:320px;}
.brand .t1{font-weight:900;letter-spacing:.5px}
.brand .t2{color:var(--muted);font-size:13px}

.feedbox{flex:0 0 auto}

.midrow{
  margin-top:12px;
  display:flex;
  align-items:flex-start;
  gap:12px;
  flex-wrap:wrap;
}

.feedbuttons{
  display:flex;
  gap:8px;
  flex-wrap:wrap;
  align-items:center;
}

.botrow{
  margin-top:12px;
  display:flex;
  align-items:center;
  justify-content:space-between;
  gap:12px;
}

.spacer{flex:1}

.rightcontrols{
  display:flex;
  gap:10px;
  align-items:center;
  flex-wrap:wrap;
  justify-content:flex-start; /* klok meer links in dit blok */
}

/* ===== Knoppen/velden ===== */
.btn{
  appearance:none;
  border:1px solid var(--line);
  color:var(--text);
  background: rgba(255,255,255,.03);
  padding:9px 11px;
  border-radius:12px;
  cursor:pointer;
  font-weight:700;
  font-size:13px;
}
.btn:hover{border-color: rgba(167,255,26,.45)}
.btn.primary{border-color: rgba(167,255,26,.45)}

.field{
  display:flex;
  gap:8px;
  align-items:center;
  border:1px solid var(--line);
  border-radius:12px;
  padding:8px 10px;
  background: rgba(255,255,255,.03);
}
.field label{font-size:12px;color:var(--muted);font-weight:700}

/* ===== Live clock ===== */
#liveClock{
  font-weight:900;
  font-size:18px;
  padding:8px 12px;
  border:1px solid var(--line);
  border-radius:999px;
  background: rgba(0,0,0,.25);
  min-width:180px;   /* iets compacter dan 220 */
  text-align:center;
}

/* ===== Nu Live inline ===== */
.nulive-inline{
  align-self:flex-start;
  margin-top:2px;
  padding:8px 10px;
  border-radius:12px;
  border:1px solid rgba(167,255,26,.22);
  background: rgba(167,255,26,.08);
  color: rgba(233,236,255,.92);
  font-weight:800;
  font-size:13px;
  max-width: 460px;
  line-height:1.25;
  flex:0 1 460px;
}
.nulive-inline .tag{
  color: var(--accent);
  font-weight:1000;
  margin-right:6px;
}

/* ===== Timeline ===== */
.timeline{
  margin-top:12px;
  height:10px;
  border-radius:999px;
  border:1px solid rgba(255,255,255,.10);
  background: rgba(0,0,0,.20);
  overflow:hidden;
}
.timeline > div{
  height:100%;
  width:0%;
  background: linear-gradient(90deg, rgba(167,255,26,.2), rgba(167,255,26,.75));
}
.smallnote{margin-top:8px;color:var(--muted);font-size:12px}

/* ===== Cards ===== */
.board{
  margin-top:14px;
  display:grid;
  grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
  gap:12px;
}
.card{
  border:1px solid var(--line);
  border-radius:16px;
  overflow:hidden;
  position:relative;
  background: linear-gradient(180deg, rgba(255,255,255,.04), rgba(255,255,255,.02));
  box-shadow: 0 12px 30px var(--shadow);
}
.card.pulse{box-shadow: 0 0 0 2px rgba(167,255,26,.30), 0 12px 30px var(--shadow);}

.hdr{
  display:flex;
  justify-content:space-between;
  align-items:flex-start;
  padding:12px;
  background: rgba(0,0,0,.18);
  border-bottom:1px solid rgba(255,255,255,.06);
  gap:10px;
}
.title{font-weight:900;line-height:1.12;letter-spacing:.2px;font-size:14px}
.score{
  font-weight:1000;
  font-size:20px;
  padding:4px 10px;
  border-radius:12px;
  border:1px solid rgba(255,255,255,.10);
  background: rgba(0,0,0,.22);
  white-space:nowrap;
}

.meta{
  padding:10px 12px;
  display:flex;
  justify-content:space-between;
  gap:10px;
  flex-wrap:wrap;
  border-bottom:1px solid rgba(255,255,255,.06);
}
.chip{
  font-size:12px;
  font-weight:800;
  padding:6px 10px;
  border-radius:999px;
  border:1px solid rgba(255,255,255,.10);
  background: rgba(0,0,0,.22);
  color:var(--muted);
}
.chip.att{display:none;color:var(--accent);}

.events{
  padding:10px 12px 12px;
  display:flex;
  flex-direction:column;
  gap:7px;
  min-height:110px;
}
.ev{
  display:flex;
  gap:10px;
  align-items:flex-start;
  font-size:13px;
  padding:6px 8px;
  border-radius:12px;
  background: rgba(0,0,0,.20);
  border:1px solid rgba(255,255,255,.06);
}
.ev .min{width:46px;flex:0 0 46px;font-weight:900;color:var(--muted);}
.ev .txt{flex:1}

.badge{
  flex:0 0 auto;
  font-size:11px;
  font-weight:1000;
  padding:3px 7px;
  border-radius:10px;
  border:1px solid rgba(255,255,255,.12);
  background: rgba(255,255,255,.04);
  color:var(--text);
  margin-left:6px;
}
.ev-goal .txt{font-weight:1000}
.ev-card_y .badge{border-color: rgba(255,208,0,.45); color: var(--warn)}
.ev-card_r .badge{border-color: rgba(255,59,59,.55); color: var(--danger)}
.ev-inj .badge{border-color: rgba(160,180,255,.45); color: #cbd7ff}
.ev-sub .badge{border-color: rgba(167,255,26,.35); color: var(--accent)}

.empty{
  margin-top:14px;
  border:1px dashed rgba(255,255,255,.18);
  border-radius:16px;
  padding:14px;
  color:var(--muted);
  background: rgba(255,255,255,.03);
}

.stadium{
  margin-top:4px;
  font-size:12px;
  color: var(--muted);
  font-weight:800;
}

  </style>
</head>
<body>
<div class="wrap">

  <div class="topbar">

  <!-- Rij 1: titel links, CSV/Laad rechts -->
  <div class="toprow">
    <div class="brand">
      <div class="t1"><?= h($leagueName) ?> — Speelronde Live - 1500 ms</div>
      <div class="t2">CSV-driven • 1–48, 15 min rust, 45–94 • toeschouwers pas na 75'</div>
    </div>

     <?php if (in_array($feedKey, $showNuLiveFeeds, true) && $nuLiveText !== ''): ?>
    <div class="nulive-inline">
      <span class="tag">Nu Live:</span> <?= h($nuLiveText) ?>
    </div>
  <?php endif; ?>

    <div class="feedbox">
      <form method="get" class="field" style="gap:10px">
        <label for="feed">CSV</label>
        <select id="feed" name="feed"
                style="background:transparent;border:0;outline:0;color:var(--text);font-weight:900;appearance:none;">
          <?php foreach ($feeds as $k => $info): ?>
            <option value="<?= h($k) ?>" <?= $k===$feedKey ? 'selected' : '' ?>>
              <?= h($info['label']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <button class="btn" type="submit" style="padding:7px 10px;border-radius:10px">Laad</button>
      </form>
    </div>
  </div>

  <!-- Rij 2: links Nu Live (alleen main), daarna feed buttons + FC Koks City -->
  <div class="midrow">
        <div class="feedbuttons">
      <?php foreach ($feeds as $k => $info): ?>
        <a class="btn <?= $k===$feedKey ? 'primary' : '' ?>"
           href="?feed=<?= h($k) ?>"
           style="text-decoration:none;display:inline-flex;align-items:center;">
          <?= h($info['label']) ?>
        </a>
      <?php endforeach; ?>

      <a class="btn" href="round_live_beker.php" style="text-decoration:none;">Beker</a>

      </div>
  </div>

  <!-- Rij 3: rechts live minuut + controls -->
  <div class="botrow">
    <div class="spacer"></div>

    <div class="rightcontrols">
      <div id="liveClock">—</div>
      <button class="btn primary" id="btnPlay">Play</button>
      <button class="btn" id="btnPause">Pauze</button>
      <button class="btn" id="btnReset">Reset</button>
    </div>
  </div>

</div>


  <?php if (!$hasData): ?>
    <div class="empty">
      <div style="font-weight:900;color:var(--text);margin-bottom:6px;">
  Geen data gevonden in <?= h(basename($csvFile)) ?> (<?= h($feedLabel) ?>)
</div>
<div>Check: delimiter is <b>;</b> en je hebt 2 kolommen: <b>Minuut_Event</b> en <b>Wedstrijd</b>.</div>
    </div>
  <?php else: ?>

    <div class="timeline"><div id="bar"></div></div>
    <div class="smallnote" id="phaseNote"></div>

    <div class="board">
      <?php foreach ($round as $i => $m): ?>
        <div class="card" id="m<?= (int)$i ?>">
          <div class="hdr">
            <div>
  <div class="title"><?= h($m["home"]) ?> <span style="color:var(--muted);font-weight:900">vs</span> <?= h($m["away"]) ?></div>
  <?php if (!empty($m['stadium'])): ?>
    <div class="stadium"><?= h($m['stadium']) ?></div>
  <?php endif; ?>
</div>
<div class="score">0 - 0</div>
          </div>

          <div class="meta">
            <div class="chip">Live feed</div>
            <div class="chip att">Toeschouwers: <?= $m["attendance"] ? (int)$m["attendance"] : 0 ?></div>
          </div>

          <div class="events"></div>
        </div>
      <?php endforeach; ?>
    </div>

    <script>
      // ===== LIVE ENGINE (opgeschoond) =====
window.ROUND_DATA = <?= json_encode($round, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) ?>;

const SPEED_MS = 1500;

let phase = "H1";     // H1, HT, H2
let liveMin = 1;
let htLeft = 15;
let isPlaying = false;

let timer = null;
const shown = new Set();

const $clock = document.querySelector("#liveClock");
const $phaseNote = document.querySelector("#phaseNote");
const $bar = document.querySelector("#bar");

function phaseLabel(){
  if (phase === "H1") return "1e helft";
  if (phase === "HT") return "Pauze";
  return "2e helft";
}

function updateClockUI(){
  if (phase === "HT") {
    $clock.textContent = `Pauze – ${htLeft} min`;
    $phaseNote.textContent = `Rust… events t/m 48' blijven zichtbaar.`;
  } else {
    const label = (phase === "H1") ? "1e helft" : "2e helft";
    $clock.textContent = `${label} – minuut ${liveMin}`;
    $phaseNote.textContent = (phase === "H1")
      ? `We spelen door tot 48' (blessuretijd). Daarna 15' pauze.`
      : `Tweede helft: 45' tot 94' (blessuretijd).`;
  }

  // progress bar blijft hetzelfde
  const total = 48 + 15 + 50; // = 113
  let done = 0;
  if (phase==="H1") done = liveMin;
  else if (phase==="HT") done = 48 + (15 - htLeft);
  else done = 48 + 15 + (liveMin - 45 + 1);

  const pct = Math.max(0, Math.min(100, (done / total) * 100));
  $bar.style.width = pct.toFixed(2) + "%";
}


function isEventVisible(minute){
  if (!isPlaying) return false;
  if (phase === "H1") return minute <= liveMin;
  if (phase === "HT") return minute <= 48;
  return minute <= liveMin; // H2
}

function escapeHtml(s){
  return String(s).replace(/[&<>"']/g, c => ({
    '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'
  }[c]));
}

function iconForType(t){
  if (t==="goal") return "⚽";
  if (t==="card_y") return "🟨";
  if (t==="card_r") return "🟥";
  if (t==="sub") return "🔁";
  if (t==="inj") return "🩹";
  return "•";
}
function badgeText(t){
  if (t==="goal") return "GOAL";
  if (t==="card_y") return "GEEL";
  if (t==="card_r") return "ROOD";
  if (t==="sub") return "WISSEL";
  if (t==="inj") return "BLES";
  return "INFO";
}

function matchScoreFromEvents(match){
  let h=0, a=0;
  for (const ev of match.events){
    if (ev.type !== "goal") continue;
    if (!isEventVisible(ev.min)) continue;
    const m = ev.text.match(/^(\d+)\s*-\s*(\d+)\s*:/);
    if (m){ h = parseInt(m[1],10); a = parseInt(m[2],10); }
  }
  return [h,a];
}

function render(){
  updateClockUI();

  const showAttendance = (phase === "H2" && liveMin >= 75);

  window.ROUND_DATA.forEach((match, idx) => {
    const card = document.querySelector(`#m${idx}`);
    const list = card.querySelector(".events");
    const scoreEl = card.querySelector(".score");
    const attEl = card.querySelector(".chip.att");

    const [h,a] = matchScoreFromEvents(match);
    scoreEl.textContent = `${h} - ${a}`;

    if (match.attendance && showAttendance){
      attEl.style.display = "inline-flex";
      attEl.textContent = `Toeschouwers: ${match.attendance}`;
    } else {
      attEl.style.display = "none";
    }

    for (const ev of match.events){
      const key = `${idx}|${ev.min}|${ev.type}|${ev.text}`;
      if (shown.has(key)) continue;

      if (isEventVisible(ev.min)){
        shown.add(key);

        const row = document.createElement("div");
        row.className = `ev ev-${ev.type}`;
        row.innerHTML =
          `<span class="min">${ev.min}'</span>` +
          `<span class="txt">${iconForType(ev.type)} ${escapeHtml(ev.text)} <span class="badge">${badgeText(ev.type)}</span></span>`;
        list.appendChild(row);

        card.classList.add("pulse");
        setTimeout(()=>card.classList.remove("pulse"), 650);
      }
    }
  });
}

function tick(){
  if (phase === "H1"){
    if (liveMin < 48) liveMin++;
    else { phase = "HT"; htLeft = 15; }
  } else if (phase === "HT"){
    htLeft--;
    if (htLeft <= 0){ phase = "H2"; liveMin = 45; }
  } else {
    if (liveMin < 94) liveMin++;
    else stop();
  }
  render();
}

// ===== controls =====
function stop(){
  isPlaying = false;
  if (timer){ clearInterval(timer); timer = null; }
  render();
}

function play(){
  stop();              // stopt vorige timer
  isPlaying = true;
  timer = setInterval(tick, SPEED_MS);
  render();
}

function reset(){
  stop();              // stopt + isPlaying=false

  phase = "H1";
  liveMin = 1;
  htLeft = 15;
  shown.clear();

  window.ROUND_DATA.forEach((_, idx)=>{
    const card = document.querySelector(`#m${idx}`);
    card.querySelector(".events").innerHTML = "";
    card.querySelector(".score").textContent = "0 - 0";
    card.querySelector(".chip.att").style.display = "none";
  });

  render();
}

// buttons
document.querySelector("#btnPlay").addEventListener("click", play);
document.querySelector("#btnPause").addEventListener("click", stop);
document.querySelector("#btnReset").addEventListener("click", reset);

// init
window.addEventListener("load", render);

    </script>

  <?php endif; ?>

</div>
</body>
</html>
