<?php
// round_live_beker.php (CSV-driven) — Beker met verlenging
declare(strict_types=1);
date_default_timezone_set('Europe/Amsterdam');

ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function normalizeWs(string $s): string {
  $s = trim($s);
  $s = preg_replace('/\s+/u', ' ', $s);
  return $s ?? trim($s);
}

function splitMatch(string $wedstrijd): array {
  $w = normalizeWs($wedstrijd);
  if (preg_match('/^(.*?)\s+vs\s+(.*?)$/iu', $w, $m)) {
    return [trim($m[1]), trim($m[2])];
  }
  return [$w, ''];
}

function detectType(string $text): string {
  $lower = mb_strtolower($text, 'UTF-8');

  if (mb_strpos($lower, 'gele kaart') !== false) return 'card_y';
  if (mb_strpos($lower, 'rode kaart') !== false) return 'card_r';
  if (mb_strpos($lower, 'wissel') !== false) return 'sub';
  if (mb_strpos($lower, 'bles') === 0 || mb_strpos($lower, 'bles.') === 0) return 'inj';
  if (preg_match('/^\d+\s*-\s*\d+\s*:/u', $text)) return 'goal';

  return 'note';
}


/**
 * CSV format:
 * Minuut_Event ; Wedstrijd
 * - Minuut_Event can be: "13e min : ...", "Aantal toeschouwers : 22537"
 */
function parseRoundCsv(string $csvFile): array {
  if (!is_file($csvFile)) return [];

  $fh = fopen($csvFile, 'r');
  if (!$fh) return [];

  $first = fgetcsv($fh, 0, ';');
  if ($first === false) { fclose($fh); return []; }

  $rows = [];
  $looksLikeHeader = false;
  if (count($first) >= 2) {
    $a = mb_strtolower(trim((string)$first[0]));
    $b = mb_strtolower(trim((string)$first[1]));
    if (str_contains($a, 'minuut') && str_contains($b, 'wedstrijd')) $looksLikeHeader = true;
  }
  if (!$looksLikeHeader) $rows[] = $first;

  while (($r = fgetcsv($fh, 0, ';')) !== false) {
    if (count($r) < 2) continue;
    $rows[] = $r;
  }
  fclose($fh);

  $byMatch = [];

  foreach ($rows as $r) {
    $minEvt = normalizeWs((string)$r[0]);
    $wed    = normalizeWs((string)$r[1]);

    if ($minEvt === '' || $wed === '') continue;
    if (!preg_match('/\s+vs\s+/iu', $wed)) continue; // alleen echte wedstrijden

    if (!isset($byMatch[$wed])) {
      [$home, $away] = splitMatch($wed);
      $byMatch[$wed] = [
        'home' => $home,
        'away' => $away,
        'attendance' => null,
        'events' => []
      ];
    }

    if (preg_match('/^Aantal toeschouwers\s*:\s*(\d+)/iu', $minEvt, $m)) {
      $byMatch[$wed]['attendance'] = (int)$m[1];
      continue;
    }

    // Event: "13e min : ..."
    if (preg_match('/^(\d+)\s*e\s*min\s*:\s*(.+)$/iu', $minEvt, $m)) {
      $min  = (int)$m[1];
      $text = trim($m[2]);
      $type = detectType($text);

      $byMatch[$wed]['events'][] = [
        'min'  => $min,
        'type' => $type,
        'text' => $text
      ];
    }
  }

  $out = array_values($byMatch);
  foreach ($out as &$match) {
    usort($match['events'], fn($a,$b)=>($a['min'] <=> $b['min']) ?: strcmp($a['text'], $b['text']));
  }
  unset($match);

  return $out;
}

// ====== DATA (Beker) ======
$csvFile   = __DIR__ . '/livefeed/raw_live_beker.csv';
$feedLabel = 'Beker';

$nuLivePath = __DIR__ . '/livefeed/DezeRonde.txt';

$leagueName = 'Panenka League'; // fallback

if (is_file($nuLivePath)) {
  $lines = file($nuLivePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  foreach ($lines as $line) {
    if (preg_match('/^\s*LeagueNaam\s*:\s*(.+)$/i', $line, $m)) {
      $leagueName = trim($m[1]);
      break;
    }
  }
}

// ====== Stadionnamen (Teams;Stadion) ======
$stadionPath = __DIR__ . '/livefeed/stadionnamen.csv';
$stadionByTeam = [];



if (is_file($stadionPath)) {
  if (($fhS = fopen($stadionPath, 'r')) !== false) {
    fgetcsv($fhS, 0, ';'); // header: Teams;Stadion
    while (($rS = fgetcsv($fhS, 0, ';')) !== false) {
      if (count($rS) < 2) continue;
      $team = normalizeWs((string)$rS[0]);
      $stad = normalizeWs((string)$rS[1]);
      if ($team === '' || $stad === '') continue;
      $stadionByTeam[$team] = $stad;
    }
    fclose($fhS);
  }
}


$round   = parseRoundCsv($csvFile);
$hasData = count($round) > 0;

// stadion koppelen (toon thuisstadion, fallback op uit)
foreach ($round as &$m) {
  $homeSt = $stadionByTeam[$m['home']] ?? '';
  $awaySt = $stadionByTeam[$m['away']] ?? '';
  $m['stadium'] = $homeSt !== '' ? $homeSt : $awaySt;
}
unset($m);


// 94 minuten -> 1500ms, +30 min verlenging => +500ms (jouw afspraak)
$speedMs = 2000;
?>
<!doctype html>
<html lang="nl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title><?= h($leagueName) ?> — Beker Live</title>
  <style>
    :root{
      --bg:#07070b; --panel:#0f1016; --line:#2a2d3a;
      --text:#e9ecff; --muted:#9aa3c7; --accent:#a7ff1a;
      --warn:#ffd000; --danger:#ff3b3b; --shadow: rgba(0,0,0,.45);
    }
    *{box-sizing:border-box}
    body{
      margin:0;
      background: radial-gradient(900px 600px at 20% 0%, rgba(167,255,26,.08), transparent 60%),
                  radial-gradient(900px 600px at 80% 10%, rgba(120,110,255,.08), transparent 60%),
                  var(--bg);
      color:var(--text);
      font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
    }
    .wrap{max-width:1200px;margin:0 auto;padding:18px 14px 32px}
    .topbar{
      display:flex; gap:12px; align-items:center; justify-content:space-between;
      padding:14px; border:1px solid var(--line); border-radius:14px;
      background: linear-gradient(180deg, rgba(255,255,255,.05), rgba(255,255,255,.02));
      box-shadow: 0 10px 30px var(--shadow);
    }
    .brand{display:flex;flex-direction:column;gap:2px}
    .brand .t1{font-weight:900;letter-spacing:.5px}
    .brand .t2{color:var(--muted); font-size:13px}
    #liveClock{
      font-weight:900;font-size:18px;padding:8px 12px;border:1px solid var(--line);
      border-radius:999px;background: rgba(0,0,0,.25);min-width:240px;text-align:center;
    }
    .controls{display:flex;gap:10px;align-items:center;flex-wrap:wrap;justify-content:flex-end}
    .btn{
      appearance:none;border:1px solid var(--line);color:var(--text);
      background: rgba(255,255,255,.03);padding:9px 11px;border-radius:12px;
      cursor:pointer;font-weight:700;font-size:13px;
      text-decoration:none;display:inline-flex;align-items:center;
    }
    .btn:hover{border-color: rgba(167,255,26,.45)}
    .btn.primary{border-color: rgba(167,255,26,.45)}
    .field{
      display:flex;gap:8px;align-items:center;border:1px solid var(--line);
      border-radius:12px;padding:8px 10px;background: rgba(255,255,255,.03);
    }
    .field label{font-size:12px;color:var(--muted);font-weight:700}
    .field input{width:90px;background:transparent;border:0;outline:0;color:var(--text);font-weight:900}

    .timeline{margin-top:12px;height:10px;border-radius:999px;border:1px solid rgba(255,255,255,.10);
      background: rgba(0,0,0,.20);overflow:hidden;}
    .timeline > div{height:100%;width:0%;background: linear-gradient(90deg, rgba(167,255,26,.2), rgba(167,255,26,.75));}
    .smallnote{margin-top:8px;color: var(--muted);font-size:12px}

    .board{margin-top:14px;display:grid;grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));gap:12px;}
    .card{
      border:1px solid var(--line);border-radius:16px;overflow:hidden;position:relative;
      background: linear-gradient(180deg, rgba(255,255,255,.04), rgba(255,255,255,.02));
      box-shadow: 0 12px 30px var(--shadow);
    }
    .card.pulse{box-shadow: 0 0 0 2px rgba(167,255,26,.30), 0 12px 30px var(--shadow);}
    .hdr{display:flex;justify-content:space-between;align-items:flex-start;padding:12px;
      background: rgba(0,0,0,.18);border-bottom:1px solid rgba(255,255,255,.06);gap:10px;}
    .title{font-weight:900;line-height:1.12;letter-spacing:.2px;font-size:14px}
    .score{font-weight:1000;font-size:20px;padding:4px 10px;border-radius:12px;border:1px solid rgba(255,255,255,.10);
      background: rgba(0,0,0,.22);white-space:nowrap;}
    .meta{padding:10px 12px;display:flex;justify-content:space-between;gap:10px;flex-wrap:wrap;border-bottom:1px solid rgba(255,255,255,.06);}
    .chip{font-size:12px;font-weight:800;padding:6px 10px;border-radius:999px;border:1px solid rgba(255,255,255,.10);
      background: rgba(0,0,0,.22);color:var(--muted);}
    .chip.att{display:none;color: var(--accent);}
    .events{padding:10px 12px 12px;display:flex;flex-direction:column;gap:7px;min-height:110px;}
    .ev{display:flex;gap:10px;align-items:flex-start;font-size:13px;padding:6px 8px;border-radius:12px;
      background: rgba(0,0,0,.20);border: 1px solid rgba(255,255,255,.06);}
    .ev .min{width:46px;flex:0 0 46px;font-weight:900;color: var(--muted);}
    .ev .txt{flex:1}
    .badge{flex:0 0 auto;font-size:11px;font-weight:1000;padding:3px 7px;border-radius:10px;border:1px solid rgba(255,255,255,.12);
      background: rgba(255,255,255,.04);color: var(--text);margin-left:6px;}
    .ev-goal .txt{font-weight:1000}
    .ev-card_y .badge{border-color: rgba(255,208,0,.45); color: var(--warn)}
    .ev-card_r .badge{border-color: rgba(255,59,59,.55); color: var(--danger)}
    .ev-inj .badge{border-color: rgba(160,180,255,.45); color: #cbd7ff}
    .ev-sub .badge{border-color: rgba(167,255,26,.35); color: var(--accent)}
    .empty{margin-top:14px;border:1px dashed rgba(255,255,255,.18);border-radius:16px;padding:14px;color: var(--muted);background: rgba(255,255,255,.03);}

    .stadium{
  margin-top:4px;
  font-size:12px;
  color: var(--muted);
  font-weight:800;
}

  </style>
</head>
<body>
<div class="wrap">

  <div class="topbar">
    <div class="brand">
      <div class="t1"><?= h($leagueName) ?> — Beker Live - 2000 ms</div>
      <div class="t2">CSV-driven • 1–48, 15 min rust, 45–94 • + verlenging • pauzes 25s en 20s</div>
    </div>

    <div id="liveClock">—</div>

    <div class="controls">
      <a class="btn" href="round_live.php">← Competitie</a>

      <button class="btn primary" id="btnPlay">Play</button>
      <button class="btn" id="btnPause">Pauze</button>
      <button class="btn" id="btnReset">Reset</button>

      
    </div>
  </div>

  <?php if (!$hasData): ?>
    <div class="empty">
      <div style="font-weight:900;color:var(--text);margin-bottom:6px;">
        Geen data gevonden in <?= h(basename($csvFile)) ?> (<?= h($feedLabel) ?>)
      </div>
      <div>Check: delimiter is <b>;</b> en je hebt 2 kolommen: <b>Minuut_Event</b> en <b>Wedstrijd</b>.</div>
    </div>
  <?php else: ?>

    <div class="timeline"><div id="bar"></div></div>
    <div class="smallnote" id="phaseNote"></div>

    <div class="board">
      <?php foreach ($round as $i => $m): ?>
        <div class="card" id="m<?= (int)$i ?>">
          <div class="hdr">
            <div>
  <div class="title"><?= h($m["home"]) ?> <span style="color:var(--muted);font-weight:900">vs</span> <?= h($m["away"]) ?></div>
  <?php if (!empty($m['stadium'])): ?>
    <div class="stadium"><?= h($m['stadium']) ?></div>
  <?php endif; ?>
</div>
<div class="score">0 - 0</div>
          </div>

          <div class="meta">
            <div class="chip">Live feed</div>
            <div class="chip att">Toeschouwers: <?= $m["attendance"] ? (int)$m["attendance"] : 0 ?></div>
          </div>

          <div class="events"></div>
        </div>
      <?php endforeach; ?>
    </div>

    <script>
      window.ROUND_DATA = <?= json_encode($round, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) ?>;

      const SPEED_MS = 2000;

      // fases: H1, HT, H2, P94, ET1, PET, ET2, END
      let phase = "H1";
      let liveMin = 1;

      let htLeft = 15;     // minuten rust
      let breakSec = 0;    // seconden-pauzes (real time)

      let timerMin = null;
      let timerSec = null;
      let isPlaying = false;


      const shown = new Set();

      const $clock = document.querySelector("#liveClock");
      const $phaseNote = document.querySelector("#phaseNote");
      const $bar = document.querySelector("#bar");

      function escapeHtml(s){
        return String(s).replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[c]));
      }
      function iconForType(t){
        if (t==="goal") return "⚽";
        if (t==="card_y") return "🟨";
        if (t==="card_r") return "🟥";
        if (t==="sub") return "🔁";
        if (t==="inj") return "🩹";
        return "•";
      }
      function badgeText(t){
        if (t==="goal") return "GOAL";
        if (t==="card_y") return "GEEL";
        if (t==="card_r") return "ROOD";
        if (t==="sub") return "WISSEL";
        if (t==="inj") return "BLES";
        return "INFO";
      }

      function phaseLabel(){
        if (phase==="H1") return "1e helft";
        if (phase==="HT") return "Pauze";
        if (phase==="H2") return "2e helft";
        if (phase==="P94") return "Wachten";
        if (phase==="ET1") return "Verlenging 1";
        if (phase==="PET") return "Wissel";
        if (phase==="ET2") return "Verlenging 2";
        return "Klaar";
      }

      function updateClockUI(){
        if (phase==="HT") {
          $clock.textContent = `${phaseLabel()} – ${htLeft} min`;
          $phaseNote.textContent = `Rust… events t/m 48' blijven zichtbaar.`;
        } else if (phase==="P94" || phase==="PET") {
          $clock.textContent = `${phaseLabel()} – ${breakSec} sec`;
          $phaseNote.textContent = (phase==="P94")
            ? `Einde reguliere speeltijd. Klaarmaken voor verlenging…`
            : `Korte pauze tussen verlengingen…`;
        } else if (phase==="END") {
          $clock.textContent = `Einde`;
          $phaseNote.textContent = `Wedstrijd afgelopen.`;
        } else {
          $clock.textContent = `${phaseLabel()} – minuut ${liveMin}`;
          if (phase==="H1") $phaseNote.textContent = `We spelen door tot 48' (blessuretijd). Daarna 15' pauze.`;
          else if (phase==="H2") $phaseNote.textContent = `Tweede helft: 45' tot 94'. Daarna 25 sec pauze, dan verlenging.`;
          else if (phase==="ET1") $phaseNote.textContent = `Verlenging: 15 minuten (95'–109').`;
          else if (phase==="ET2") $phaseNote.textContent = `Verlenging: laatste 15 minuten (110'–124').`;
        }

        // timeline total: 48 + 15 + 50 + 30 = 143 "minuten-stappen"
        const total = 48 + 15 + 50 + 30;
        let done = 0;

        if (phase==="H1") done = liveMin;
        else if (phase==="HT") done = 48 + (15 - htLeft);
        else if (phase==="H2") done = 48 + 15 + (liveMin - 45 + 1);
        else if (phase==="P94") done = 48 + 15 + 50;
        else if (phase==="ET1") done = 48 + 15 + 50 + (liveMin - 95 + 1);
        else if (phase==="PET") done = 48 + 15 + 50 + 15;
        else if (phase==="ET2") done = 48 + 15 + 50 + 15 + (liveMin - 110 + 1);
        else done = total;

        const pct = Math.max(0, Math.min(100, (done/total)*100));
        $bar.style.width = pct.toFixed(2) + "%";
      }

      function isEventVisible(minute){
        if (!isPlaying) return false;
        if (phase==="H1") return minute <= liveMin;
        if (phase==="HT") return minute <= 48;
        if (phase==="H2") return minute <= liveMin;       // 45..94
        if (phase==="P94") return minute <= 94;
        if (phase==="ET1") return minute <= liveMin;       // 95..109
        if (phase==="PET") return minute <= 109;
        if (phase==="ET2") return minute <= liveMin;       // 110..124
        return true;
      }

      function matchScoreFromEvents(match){
        let h=0,a=0;
        for (const ev of match.events){
          if (ev.type !== "goal") continue;
          if (!isEventVisible(ev.min)) continue;
          const m = ev.text.match(/^(\d+)\s*-\s*(\d+)\s*:/);
          if (m){ h=parseInt(m[1],10); a=parseInt(m[2],10); }
        }
        return [h,a];
      }

      function stopAll(){
        if (timerMin){ clearInterval(timerMin); timerMin=null; }
        if (timerSec){ clearInterval(timerSec); timerSec=null; }
      }

      function startMinuteTimer(){
        stopAll();
        timerMin = setInterval(tickMinute, SPEED_MS);
      }

      function startSecondBreak(seconds){
        stopAll();
        breakSec = seconds;
        render();
        timerSec = setInterval(()=>{
          breakSec--;
          render();
          if (breakSec <= 0){
            clearInterval(timerSec);
            timerSec = null;

            if (phase==="P94"){ phase="ET1"; liveMin=95; startMinuteTimer(); }
            else if (phase==="PET"){ phase="ET2"; liveMin=110; startMinuteTimer(); }
          }
        }, 1000);
      }

      function tickMinute(){
        if (phase==="H1"){
          if (liveMin < 48) liveMin++;
          else { phase="HT"; htLeft=15; }
        }
        else if (phase==="HT"){
          htLeft--;
          if (htLeft <= 0){ phase="H2"; liveMin=45; }
        }
        else if (phase==="H2"){
          if (liveMin < 94) liveMin++;
          else { phase="P94"; startSecondBreak(25); return; }
        }
        else if (phase==="ET1"){
          if (liveMin < 109) liveMin++;
          else { phase="PET"; startSecondBreak(20); return; }
        }
        else if (phase==="ET2"){
          if (liveMin < 124) liveMin++;
          else { phase="END"; stopAll(); }
        }

        render();
      }

      function render(){
        updateClockUI();

        const showAttendance = (
          (phase==="H2" || phase==="P94" || phase==="ET1" || phase==="PET" || phase==="ET2" || phase==="END")
          && liveMin >= 75
        );

        window.ROUND_DATA.forEach((match, idx) => {
          const card = document.querySelector(`#m${idx}`);
          const list = card.querySelector(".events");
          const scoreEl = card.querySelector(".score");
          const attEl = card.querySelector(".chip.att");

          const [h,a] = matchScoreFromEvents(match);
          scoreEl.textContent = `${h} - ${a}`;

          if (match.attendance && showAttendance){
            attEl.style.display = "inline-flex";
            attEl.textContent = `Toeschouwers: ${match.attendance}`;
          } else {
            attEl.style.display = "none";
          }

          for (const ev of match.events){
            const key = `${idx}|${ev.min}|${ev.type}|${ev.text}`;
            if (shown.has(key)) continue;

            if (isEventVisible(ev.min)){
              shown.add(key);

              const row = document.createElement("div");
              row.className = `ev ev-${ev.type}`;
              row.innerHTML =
                `<span class="min">${ev.min}'</span>` +
                `<span class="txt">${iconForType(ev.type)} ${escapeHtml(ev.text)} <span class="badge">${badgeText(ev.type)}</span></span>`;
              list.appendChild(row);

              card.classList.add("pulse");
              setTimeout(()=>card.classList.remove("pulse"), 650);
            }
          }
        });
      }
      function play(){
  isPlaying = true;
  startMinuteTimer();
  render();
}

      function stop(){ stopAll(); }

      function reset(){
        stopAll();
        isPlaying = false;
        phase="H1"; liveMin=1; htLeft=15; breakSec=0;
        shown.clear();

        window.ROUND_DATA.forEach((_, idx)=>{
          const card = document.querySelector(`#m${idx}`);
          card.querySelector(".events").innerHTML = "";
          card.querySelector(".score").textContent = "0 - 0";
          card.querySelector(".chip.att").style.display = "none";
        });

        render();
      }

      document.querySelector("#btnPlay").addEventListener("click", play);
      document.querySelector("#btnPause").addEventListener("click", stop);
      document.querySelector("#btnReset").addEventListener("click", reset);
      
      window.addEventListener("load", render);
    </script>

  <?php endif; ?>

</div>
</body>
</html>
